/* crc32.c -- compute the CRC-32 of a data stream
 * Copyright (C) 1995-2002 Mark Adler
 * For conditions of distribution and use, see copyright notice in zlib.h
 *
 * Thanks to Rodney Brown <rbrown64@csc.com.au> for his contribution of faster
 * CRC methods: exclusive-oring 32 bits of data at a time, and pre-computing
 * tables for updating the shift register in one step with three exclusive-ors
 * instead of four steps with four exclusive-ors.  This results in more than a
 * factor-of-two increase in speed on a Power PC using gcc -O3.
 */

/* @(#) $Id$ */
#ifdef WIN32
#ifndef __STDC__
#define __STDC__ (1)
#endif
#endif

#include "zutil.h"

#define local static

/* Find a four-byte integer type for crc32_little() and crc32_big(). */
#ifndef NOBYFOUR
#  ifdef __STDC__   /* need ANSI C limits.h to determine sizes */
#    include <limits.h>
#    define BYFOUR
#    if (UINT_MAX == 4294967295)
       typedef unsigned int u4;
#    elif (ULONG_MAX == 4294967295)
       typedef unsigned long u4;
#    elif (USHRT_MAX == 4294967295)
       typedef unsigned short u4;
#    else
#      undef BYFOUR /* can't find a four-byte integer type! */
#    endif
#  endif /* __STDC__ */
#endif /* !NOBYFOUR */

/* Definitions for doing the crc four data bytes at a time. */
#ifdef BYFOUR
   typedef u4 FAR u4f;
#  define REV(w) (((w)>>24)+(((w)>>8)&0xff00)+ \
                (((w)&0xff00)<<8)+(((w)&0xff)<<24))
   local uLong crc32_little OF((uLong, const Bytef *, uInt));
   local uLong crc32_big OF((uLong, const Bytef *, uInt));
#  define TBLS 8
#else
#  define TBLS 1
#endif /* BYFOUR */

#ifdef DYNAMIC_CRC_TABLE

local int crc_table_empty = 1;
local uLongf crc_table[TBLS][256];
local void make_crc_table OF((void));
#ifdef DEBUG
   local void write_table OF((FILE *, const uLongf *));
#endif /* DEBUG */

/*
  Generate tables for a byte-wise 32-bit CRC calculation on the polynomial:
  x^32+x^26+x^23+x^22+x^16+x^12+x^11+x^10+x^8+x^7+x^5+x^4+x^2+x+1.

  Polynomials over GF(2) are represented in binary, one bit per coefficient,
  with the lowest powers in the most significant bit.  Then adding polynomials
  is just exclusive-or, and multiplying a polynomial by x is a right shift by
  one.  If we call the above polynomial p, and represent a byte as the
  polynomial q, also with the lowest power in the most significant bit (so the
  byte 0xb1 is the polynomial x^7+x^3+x+1), then the CRC is (q*x^32) mod p,
  where a mod b means the remainder after dividing a by b.

  This calculation is done using the shift-register method of multiplying and
  taking the remainder.  The register is initialized to zero, and for each
  incoming bit, x^32 is added mod p to the register if the bit is a one (where
  x^32 mod p is p+x^32 = x^26+...+1), and the register is multiplied mod p by
  x (which is shifting right by one and adding x^32 mod p if the bit shifted
  out is a one).  We start with the highest power (least significant bit) of
  q and repeat for all eight bits of q.

  The first table is simply the CRC of all possible eight bit values.  This is
  all the information needed to generate CRCs on data a byte at a time for all
  combinations of CRC register values and incoming bytes.  The remaining tables
  allow for word-at-a-time CRC calculation for both big-endian and little-
  endian machines, where a word is four bytes.
*/
local void make_crc_table()
{
    uLong c;
    int n, k;
    uLong poly;            /* polynomial exclusive-or pattern */
    /* terms of polynomial defining this crc (except x^32): */
    static const Byte p[] = {0,1,2,4,5,7,8,10,11,12,16,22,23,26};

    /* make exclusive-or pattern from polynomial (0xedb88320L) */
    poly = 0L;
    for (n = 0; n < sizeof(p)/sizeof(Byte); n++)
        poly |= 1L << (31 - p[n]);

    /* generate a crc for every 8-bit value */
    for (n = 0; n < 256; n++)
    {
        c = (uLong)n;
        for (k = 0; k < 8; k++)
            c = c & 1 ? poly ^ (c >> 1) : c >> 1;
        crc_table[0][n] = c;
    }

#ifdef BYFOUR
    /* generate crc for each value followed by one, two, and three zeros, and
       then the byte reversal of those as well as the first table */
    for (n = 0; n < 256; n++)
    {
        c = crc_table[0][n];
        crc_table[4][n] = REV(c);
        for (k = 1; k < 4; k++)
        {
            c = crc_table[0][c & 0xff] ^ (c >> 8);
            crc_table[k][n] = c;
            crc_table[k + 4][n] = REV(c);
        }
    }
#endif /* BYFOUR */

  crc_table_empty = 0;

#ifdef DEBUG
    /* write out CRC tables to crc32.h */
    {
        FILE *out;

        out = fopen("crc32.h", "w");
        if (out == NULL) return;
        fprintf(out, "/* crc32.h -- tables for rapid CRC calculation\n");
        fprintf(out, " * Generated automatically by crc32.c\n */\n\n");
        fprintf(out, "local const uLongf crc_table[TBLS][256] =\n{\n  {\n");
        write_table(out, crc_table[0]);
#  ifdef BYFOUR
        fprintf(out, "#ifdef BYFOUR\n");
        for (k = 1; k < 8; k++)
        {
            fprintf(out, "  },\n  {\n");
            write_table(out, crc_table[k]);
        }
        fprintf(out, "#endif\n");
#  endif /* BYFOUR */
        fprintf(out, "  }\n};\n");
        fclose(out);
    }
#endif /* DEBUG */
}

#ifdef DEBUG
local void write_table(out, table)
    FILE *out;
    const uLongf *table;
{
    int n;

    for (n = 0; n < 256; n++)
        fprintf(out, "%s0x%08lxL%s", n % 5 ? "" : "    ", table[n],
                n == 255 ? "\n" : (n % 5 == 4 ? ",\n" : ", "));
}
#endif /* DEBUG */

#else /* !DYNAMIC_CRC_TABLE */
/* ========================================================================
 * Tables of CRC-32s of all single-byte values, made by make_crc_table().
 */
#include "crc32.h"
#endif /* DYNAMIC_CRC_TABLE */

/* =========================================================================
 * This function can be used by asm versions of crc32()
 */
const uLongf * ZEXPORT get_crc_table()
{
#ifdef DYNAMIC_CRC_TABLE
  if (crc_table_empty) make_crc_table();
#endif /* DYNAMIC_CRC_TABLE */
  return (const uLongf *)crc_table;
}

/* ========================================================================= */
#define DO1 crc = crc_table[0][((int)crc ^ (*buf++)) & 0xff] ^ (crc >> 8)
#define DO8 DO1; DO1; DO1; DO1; DO1; DO1; DO1; DO1

/* ========================================================================= */
uLong ZEXPORT crc32_newzl(crc, buf, len)
    uLong crc;
    const Bytef *buf;
    uInt len;
{
    if (buf == Z_NULL) return 0L;

#ifdef DYNAMIC_CRC_TABLE
    if (crc_table_empty)
        make_crc_table();
#endif /* DYNAMIC_CRC_TABLE */

#ifdef BYFOUR
    {
        u4 endian;

        endian = 1;
        if (*((Byte *)(&endian)))
            return crc32_little(crc, buf, len);
        else
            return crc32_big(crc, buf, len);
    }
#else /* !BYFOUR */
    crc = crc ^ 0xffffffffL;
    while (len >= 8)
    {
        DO8;
        len -= 8;
    }
    if (len) do {
        DO1;
    } while (--len);
    return crc ^ 0xffffffffL;
#endif /* BYFOUR */
}

#ifdef BYFOUR

/* ========================================================================= */
#define DOLIT4 c ^= *buf4++; \
        c = crc_table[3][c & 0xff] ^ crc_table[2][(c >> 8) & 0xff] ^ \
            crc_table[1][(c >> 16) & 0xff] ^ crc_table[0][c >> 24]

#define DOLIT16 DOLIT4; DOLIT4; DOLIT4; DOLIT4

/* ========================================================================= */
local uLong crc32_little(crc, buf, len)
    uLong crc;
    const Bytef *buf;
    uInt len;
{
    register u4 c;
    register const u4f *buf4;

    c = (u4)crc;
    c = ~c;
    while (len && ((int)buf & 3))
    {
        c = crc_table[0][(c ^ *buf++) & 0xff] ^ (c >> 8);
        len--;
    }

    buf4 = (const u4f *)buf;
    while (len >= 64)
    {
        DOLIT16;
        DOLIT16;
        DOLIT16;
        DOLIT16;
        len -= 64;
    }
    while (len >= 4)
    {
        DOLIT4;
        len -= 4;
    }
    buf = (const Bytef *)buf4;

    if (len) do {
        c = crc_table[0][(c ^ *buf++) & 0xff] ^ (c >> 8);
    } while (--len);
    c = ~c;
    return (uLong)c;
}

/* ========================================================================= */
#define DOBIG4 c ^= *buf4++; \
        c = crc_table[4][c & 0xff] ^ crc_table[5][(c >> 8) & 0xff] ^ \
            crc_table[6][(c >> 16) & 0xff] ^ crc_table[7][c >> 24]
#define DOBIG16 DOBIG4; DOBIG4; DOBIG4; DOBIG4

/* ========================================================================= */
local uLong crc32_big(crc, buf, len)
    uLong crc;
    const Bytef *buf;
    uInt len;
{
    register u4 c;
    register const u4f *buf4;

    c = REV((u4)crc);
    c = ~c;
    while (len && ((int)buf & 3))
    {
        c = crc_table[4][(c >> 24) ^ *buf++] ^ (c << 8);
        len--;
    }

    buf4 = (const u4f *)buf;
    while (len >= 64)
    {
        DOBIG16;
        DOBIG16;
        DOBIG16;
        DOBIG16;
        len -= 64;
    }
    while (len >= 4)
    {
        DOBIG4;
        len -= 4;
    }
    buf = (const Bytef *)buf4;

    if (len) do {
        c = crc_table[4][(c >> 24) ^ *buf++] ^ (c << 8);
    } while (--len);
    c = ~c;
    return (uLong)(REV(c));
}

#endif /* BYFOUR */
